/*
 * cPbcastView.cpp
 * 
 * Class implementation for a pbcast view.
 *
 * Authors:  Vicky & Ted
 */
#include "cPbcastView.h"
#include "cPbcastNode.h"
#include "Endpoint/cEndpoint.h"
#include <math.h>
#include <assert.h>
#include <stdlib.h>

#ifndef NULL
	#define NULL 0
#endif

/*
 * cPbcastView::cPbcastView()
 *
 * Purpose:	Creates a new view.
 * IN:		initialCap	-> The starting size of the queue.
 *			capInc		-> How much to grow the queue by every time.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is set up.
 * Throws:	-
 * Return:	-
 */
cPbcastView::cPbcastView(unsigned int initialCap, unsigned int capInc)
{
	mSize			= initialCap;
	mInsertPtr		= 0;
	mCapIncrement   = capInc;

	// Alloc queue space.
	mQueue = new cPbcastNode*[mSize];
	if(!mQueue)
	{
		//throw cQueueException("<cSortedMsgQueue::cSortedMsgQueue>: Unable to alloc new queue.");
	}
	for(unsigned int i = 0; i < mSize; i++)
	{
		mQueue[i] = NULL;
	}
}

/*
 * cPbcastView::~cPbcastView()
 *
 * Purpose:	Destroy the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleaned up.
 * Return:	-
 */
cPbcastView::~cPbcastView()
{
	if(mQueue) 
	{ 
		Clear();
		delete mQueue; 
	}
}


/*
 * cPbcastView::Delete()
 *
 * Purpose:	Deletes the given node from the view.
 * IN:		node		-> the node to delete.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue shrinks by at least one.
 * Return:	true if success, else false.
 */
bool cPbcastView::Delete(cPbcastNode* node)
{
	return Delete(node->mEndpoint);
}

/*
 * cPbcastView::Delete()
 *
 * Purpose:	Deletes the given node from the view.
 * IN:		ep	-> The endpoint to delete.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue shrinks by at least one.
 * Return:	true if success, else false.
 */
bool cPbcastView::Delete(cEndpoint* ep)
{
	if(mInsertPtr)
	{
		int	location = _BinSearch(ep, 0, mInsertPtr-1);
		if(location == -1)
		{
			return false;	// not present
		}
		if(mQueue[location])
		{
			mQueue[location]->Release();
		}

		// Swap elements over to fill hole
		location = location + 1;
		while(location < (int)mInsertPtr)
		{
			mQueue[location-1] = mQueue[location];
			location++;
		}
		mInsertPtr--;
		mQueue[mInsertPtr] = NULL;
		return true;
	}
	return false;
}

/*
 * cPbcastView::_Grow()
 *
 * Purpose:	Attempts to grow the queue by at least one, at most mCapIncrement
 * IN:		-
 * OUT:		-
 * Cond:	The queue is full!
 * PostCnd:	The queue is grown by at least one, at most mCapIncrement elements.
 * Return:	true if success, else false.
 */ 
bool cPbcastView::_Grow()
{
	if(mInsertPtr == mSize)
	{
		// We need to expand the queue.
		cPbcastNode** newQ = new cPbcastNode*[mSize+mCapIncrement];
		if(!newQ)
		{
			return false;
		}

		// The following may be faster using memcpy()...
		for(unsigned int i = 0; i < mSize; i++)
		{
			newQ[i] = mQueue[i];
		}
		for(i = mSize; i < mSize+mCapIncrement; i++)
		{
			newQ[i] = NULL;
		}
		mSize = mSize + mCapIncrement;
		delete mQueue;
		mQueue = newQ;
	}
 	return true;
}

/*
 * cPbcastView::Insert()
 *
 * Purpose:	Attempts to insert the node, but only if it is not in the view.
 * IN:		node	-> The node to insert.
 * OUT:		-
 * Cond:	The node is not in the queue.
 * PostCnd:	The node is placed in the queue.
 * Return:	true if success, else false.
 */ 
bool cPbcastView::Insert(cPbcastNode* node)
{
	bool	retVal;
	int		searchResult;

	// See if we need to replace.
	if(mInsertPtr)
	{
		searchResult = _BinSearch(node->mEndpoint, 0, mInsertPtr-1);
		if(searchResult != -1)
		{
			return true;	// It was already in there, but that is success.
		}
	}

	// Inserting brand new element!
	if(mInsertPtr == mSize)
	{
		if(!_Grow())
		{
			return false;
		}
	}

	mQueue[mInsertPtr] = node;
	mQueue[mInsertPtr]->AddRef();

	// Sort it
	_InsertSort();
	mInsertPtr = mInsertPtr+1;	// Not a circular queue!
	retVal = true;

	return retVal;
}

/*
 * cSubnetTree::_InsertSort()
 *
 * Purpose:	Keeps the sorted property of the queue, assuming new element is placed
 *			at mInsertPtr
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a should be delivered before b, else false.
 */ 
bool cPbcastView::_InsertSort()
{
	unsigned int	current, prev;
	cPbcastNode*	temp;

	current = prev = mInsertPtr;
	while(current != 0)
	{
		prev = prev-1;		// No wrap around!

		// Make sure that prev is not a hole!
		if(mQueue[prev])
		{
			if(!_Before(mQueue[current]->mEndpoint, mQueue[prev]->mEndpoint))
			{
				return true;		// No need to swap anymore
			}
		}

		// Swap (into a hole if one exists)
		temp = mQueue[current];
		mQueue[current] = mQueue[prev];
		mQueue[prev] = temp;

		current = prev;
	}
	return true;
}

/*
 * cPbcastView::_Before()
 *
 * Purpose:	Returns true if a is before b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is before b, else false.
 */ 
bool cPbcastView::_Before(cEndpoint* a, cEndpoint* b)
{
	return a->LessThan(b);
}

/*
 * cPbcastView::_Equal()
 *
 * Purpose:	Returns true if a refers to same subnet as b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is the same as b
 */ 
bool cPbcastView::_Equal(cEndpoint* a, cEndpoint* b)
{
	return a->Equals(b);
}

/*
 * cPbcastView::FindEpOnSubnet()
 *
 * Purpose:	Finds the first endpoint with on the given subnet.
 * IN:		ep		-> basis for search.
 * OUT:		outEp	-> the result of the search.
 * Cond:	
 * PostCnd:	
 * Return:	true if found, else false.
 */
bool cPbcastView::FindEpOnSubnet(cEndpoint* ep, cEndpoint** outEp)
{
	for(unsigned int i = 0; i < mInsertPtr; i++)
	{
		if(mQueue[i]->mEndpoint->IsSameSubnet(ep))
		{
			*outEp = mQueue[i]->mEndpoint;
			return true;
		}
	}
	return false;
}

/*
 * cPbcastView::Find()
 *
 * Purpose:	Searches for the element and returns its index.
 * IN:		ep		-> basis for search
 * OUT:		result	-> the node, if it is found.
 * Cond:	
 * PostCnd:	
 * Return:	true if found, else false.
 */
bool cPbcastView::Find(cEndpoint* ep, cPbcastNode** result)
{
	int retVal;

	if(!mInsertPtr)
	{
		return false;
	}

	retVal = _BinSearch(ep, 0, mInsertPtr-1);
	if(retVal != -1)
	{
		*result = mQueue[retVal];
	}
	return (retVal != -1);
}

/*
 * cPbcastView::_BinSearch()
 *
 * Purpose:	Performs binary search, looking for a match.
 * NOTE:	The input left/right can NOT be wrap around values.
 * IN:		ep			-> The endpoint to look for.
 *			left		-> The left bounds of the array (start w/ 0)
 *			right		-> The right bounds of the array (start w/ insert ptr-1)
 * OUT:		-
 * Cond:	The endpoint is in the queue.
 * PostCnd:	-
 * Return:	-1 if not found, else the index in mQueue of the SubnetInfo
 */
int cPbcastView::_BinSearch(cEndpoint* ep, int left, int right)
{
	int pivot;

	while(1)
	{
		if(left > right)
		{
			return -1;
		}
		else
		{
			pivot = (right-left) / 2;	// Just how much pivot advances from left, actually.
		}
		pivot = (left + pivot);	

		// See where to search next
		if(_Before(ep, mQueue[pivot]->mEndpoint))
		{
			right = pivot-1;
		}
		else if( _Equal(ep, mQueue[pivot]->mEndpoint) )
		{
			// Found it!
			return pivot;
		}
		else
		{
			// Search from pivot+1 to right
			left = pivot+1;
		}
	}
}

/*
 * cPbcastView::GetRandomNode()
 *
 * Purpose:	Returns a random node within the current view (used for choosing gossip subset).
 * IN:		-
 * OUT:		-
 * Cond:	There must be at least one node in the queue.
 * PostCnd:	-
 * Return:	A random pbcast node from the queue, or NULL if there are none.
 */
cPbcastNode* cPbcastView::GetRandomNode()
{
	int	chosen = rand();
	chosen = chosen % GetNumElements();
	return mQueue[chosen];
}

/*
 * cPbcastView::Clear()
 *
 * Purpose:	Clears out the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is reset and cleared.
 * Return:	true if success, false otherwise.
 */
bool cPbcastView::Clear()
{
	for(unsigned int i = 0; i < mSize; i++)
	{
		if(mQueue[i])
		{
			mQueue[i]->Release();
		}
	}
	mInsertPtr   = 0;
	return true;
}

void cPbcastView::UpdateDigest()
{
	unsigned int hash;
	cViewIterator* vIter = GetViewIterator();
	cEndpoint*	   ep;
	MD5Context	context;

	MD5Init(&context);
	while(!vIter->Done())
	{
		ep = vIter->GetMember();
		hash = ep->HashCode();
		MD5Update(&context, (unsigned char *)&hash, sizeof(hash));
		vIter->GetNext();
	}
	MD5Final(mDigest, &context);
}

bool cPbcastView::GetDigest(unsigned char** digest, int* digestSize)
{
	*digest = &mDigest[0];
	*digestSize = DIGEST_SIZE;
	return true;
}

// View iterator
bool		 cPbcastViewIterator::Done()			{ return (mIndex == mView->mInsertPtr); }
cEndpoint*	 cPbcastViewIterator::GetMember()		{ return mView->mQueue[mIndex]->mEndpoint; }
bool		 cPbcastViewIterator::GetNext()			{ mIndex++; return true; }
unsigned int cPbcastViewIterator::GetNumMembers()	{ return mView->GetNumElements(); }

// Node iterator
bool		 cPbcastNodeIterator::Done()			 { return (mIndex == mView->mInsertPtr); }
cObject*	 cPbcastNodeIterator::GetData()			{ return mView->mQueue[mIndex]; }
bool		 cPbcastNodeIterator::GetNext()			{ mIndex++; return true; }
bool		 cPbcastNodeIterator::DeleteCurrent()	{ return mView->Delete(mView->mQueue[mIndex]); }